//******************************************************************************
//   MSP-FET430P140 Demo - SMBus Slave Writes to MSP430 Master with PEC
//
//   Description: This demo connects two MSP430's via the I2C bus.  The master 
//   reads from the slave. This is the slave code. The master code is called
//   fet140_i2c_SMB_mstr_PEC.c. The TX data begins at 1 and is incremented 
//   each time it is sent. PEC is appended with each byte
//   The TXRDYIFG interrupt is used to know when to TX.
//   ACLK = n/a, MCLK = SMCLK = I2CIN = DCO ~ 800kHz
//  //* MSP430F169 Device Required *//
//   
//                                 /|\  /|\
//                  MSP430F169     10k  10k     MSP430F169
//                    slave         |    |        master           
//              -----------------|  |    |  ----------------- 
//             |             P3.1|<-|-----+>|P3.1            |
//             |                 |  |      |             P1.0|-->LED
//             |                 |  |      |                 |
//             |             P3.3|<-+----->|P3.3             |
//             |                 |         |                 |
//
//
//  H. Grewal
//  Texas Instruments Inc.
//  Feb 2005
//  Built with IAR Embedded Workbench Version: 3.21A
//******************************************************************************

#include  <msp430x16x.h>
#define CRC_MSG_SIZE	    2
#define CRC16_POLY	    0x07
#define CRC16_INIT_REM      0x0
typedef enum {
             SMBS_MISC               = (unsigned char) 0x48,
             SMBS_DEVICE_DFLT        = (unsigned char) 0x61,
             SMBS_ALERT_RESPONSE     = (unsigned char) 0x0C,
             SMBS_HOST               = (unsigned char) 0x08,
             SMBS_10BIT_ADD          = (unsigned char) 0x78,
             SMBS_DFLT_ADD           = (unsigned char) 0x37,
             SMBS_ACCESS_HOST        = (unsigned char) 0x28,             
             SMBS_CBUS               = (unsigned char) 0x01,
             SMBS_GCAL               = (unsigned char) 0x00
             }SMBS_Address_t;               
/*---------------------------------------------------------------------------*/	 

void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add);

unsigned char crc8MakeBitwise(unsigned char crc, unsigned short poly, unsigned char *pmsg, unsigned int msg_size);

unsigned int length;
unsigned char crc8;
unsigned char msg[CRC_MSG_SIZE] = {0};              

char TXData = 1;

void main (void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT
  P3SEL |= 0x0A;                            // Select I2C pins
  P1DIR |= 0xFF;                            // P1.0 output
                                            // unused ports to output
  P1OUT =  0;
  P2DIR =  0xFF;
  P3DIR =  0xFF;
  P4DIR =  0xFF;
  P5DIR =  0xFF;
  P6DIR =  0xFF; 
  SMBS_Init (SMBS_MISC,0x48);               // Own address configured as 48h

  _BIS_SR(LPM0_bits + GIE);                 // Enter LPM0 w/ interrupt
}

void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add)
{
  U0CTL |= I2C + SYNC;                      // Recommended init procedure
  U0CTL &= ~I2CEN;                          // Recommended init procedure
  I2CTCTL |= I2CSSEL1;                      // SMCLK
  if (Add_Param == SMBS_MISC)
  {
      I2COA = Slave_Add;                    // Own Address is 048h
  } 
  else
      I2COA = (unsigned char) Add_Param ;
  
  I2CIE = TXRDYIE;                          // Enable TXRDYIFG interrupt
  U0CTL |= I2CEN;                           // Enable I2C
}


unsigned char crc8MakeBitwise(unsigned char crc, unsigned short poly, unsigned char *pmsg, unsigned int msg_size)
{
    unsigned int i, j, carry;
    unsigned char mg;

    crc ^= *pmsg++;

    for(i = 0 ; i < msg_size ; i ++)
    {
        mg = *pmsg++;

  for(j = 0 ; j < 8 ; j++)
        {
   carry = crc & 0x80;
   crc = (crc << 1) | (mg >> 7);
   if(carry) crc ^= poly;
   mg <<= 1;
        }
    }
   
    return(crc);
}       

// Common ISR for I2C Module
#pragma vector=USART0TX_VECTOR
__interrupt void I2C_ISR(void)
{
 
 switch( I2CIV )
 {
   static unsigned int count = 0;           // Counter to determine Byte#
   case  2: break;                          // Arbitration lost
   case  4: break;                          // No Acknowledge
   case  6: break;                          // Own Address
   case  8: break;                          // Register Access Ready
   case 10: break;                          // Receive Ready
   case 12:                                 // Transmit Ready
   
   if (count == 0)
   {
    I2CDRB = TXData;                        // Load I2CDRB and increment    
    msg[0] = I2COA;
    msg[1] = TXData;
    TXData++;
    count ++;
   }
   else
   {
    crc8 = crc8MakeBitwise(CRC16_INIT_REM, CRC16_POLY, msg, CRC_MSG_SIZE);
    I2CDRB = crc8;
    count = 0;
   }
     break;
   case 14: break;                          // General Call
   case 16: break;                          // Start Condition
 }
}


